#ifndef LLF_OTF_H
#define LLF_OTF_H

/*
* All the includes that are needed for code using this module to
* compile correctly should be #included here.
*/

#include "CE2_public.h"
#include "CE2_OTF.h"
#include "CE2_AES.h"
#include "CE2_HASH.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
*  Object name     :  LLF_OTF.h
*  State           :  %state%
*  Creation date   :  28.03.2007
*  Last modified   :  %modify_time%
*/
/** @file
* \brief contains declarations of low level functions for OTF.
*
*  \version LLF_OTF.h#1:cinc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd. All rights reserved.
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

  /**
  ****************************************************************
  * Function Name: 
  *  LLF_AES2AES2HASH_OTF
  *
  * Inputs:
  * @param FirstIVCounter_ptr [in] - This parameter is the buffer of the IV or counters on mode CTR for the first key
  *                                  used in the chain.
  *                                  In ECB, XCBC, CMAC mode this parameter is not used.
  *                                  In CBC and MAC modes this parameter should contain the IV values.
  *                                  In CTR mode this value should contain the init counter.
  *                                  In XCBC and CMAC modes it may be NULL
  *  @param FirstKey_ptr [in] - A pointer to the user's key buffer for the first key
  *                                  used in the chain.
  *  @param FirstKeySize [in] - The size of the KEY used by the AES: 128, 192 or 256 bits, 
  *                             as defined in the enum for the first key used in the chain.
  *  @param FirstEncryptDecryptFlag [in] - A flag specifying whether the AES should perform an 
  *                                       Encrypt operation (0) or a Decrypt operation (1). 
  *                                       In XCBC and CMAC modes it must be 0 .
  *                                       for the first key used in the chain.
  *  @param FirstOperationMode [in] - The operation mode: ECB, CBC, MAC, CTR, XCBC (PRF and 96), CMAC.
  *                                   for the first key used in the chain.
  * @param SecondIVCounter_ptr [in] - This parameter is the buffer of the IV or 
  *                                    counters on mode CTR for the first key used in the chain.
  *                                    In ECB, XCBC, CMAC mode this parameter is not used.
  *                                    In CBC and MAC modes this parameter should contain the IV values.
  *                                    In CTR mode this value should contain the init counter.
  *                                    In XCBC and CMAC modes it may be NULL
  *                                    In case second key is not used it may be NULL
  *  @param SecondKey_ptr [in] - A pointer to the user's key buffer.
  *                               for the Second key used in the chain.
  *                               In case the second key is not used this should be NULL!           
  *  @param SecondKeySize [in] - The size of the KEY used by the AES: 128, 192 or 256 bits, 
  *                               as defined in the enum.
  *                               for the first key used in the chain.
  *                               In case second key is not used this value is ignored.
  *  @param SecondEncryptDecryptFlag [in] - A flag specifying whether the AES should perform an 
  *                                          Encrypt operation (0) or a Decrypt operation (1). 
  *                                          In XCBC and CMAC modes it must be 0.
  *                                          for the first key used in the chain.
  *                                          In case second key is not used this value is ignored.
  *  @param SecondOperationMode [in] - The operation mode: ECB, CBC, MAC, CTR, XCBC (PRF and 96), CMAC.
  *                                     for the first key used in the chain.
  *                                     In case second key is not used this value is ignored.                                         
  * @param HASHChainMode, [in] - The Chain mode - Hash according to enum:
  *                              1. Hash the plain text 
  *                              2. Hash the cipher resulted from the first key
  *                              2. Hash the cipher text resulted
  *                              3. Do not use HASH
  * @param HASHOperationMode [in] - The operation mode - this parameter is ignored in case the chain mode is - No hash
  *                               This parameter is ignored in case the chain mode is - No hash
  *  @param DataIn_ptr [in] - The pointer to the buffer of the input data to the AES chain. 
  *               The pointer's value does not need to be word-aligned.
  *  @param DataInSize [in] - The size of the input data (must be not 0 and must be multiple 
  *               of 16 bytes, besides XCBC and CMAC).
  *  @param DataOut_ptr [out] - The pointer to the buffer of the output data from the AES chain. 
  *                The pointer does not need to be aligned to 32 bits. 
  *                On XCBC and CMAC modes it may be NULL. 
  *                The pointer's value does not need to be word-aligned. 
  * @param HashResultBuff [out] - A pointer to the target buffer where the 
  *                               HASH result stored in the context is loaded to.
  *                               This parameter is ignored in case the chain mode is - No hash
  *
  * Outputs: @returns \b
  *  CE2Error_t  
  *  - comatible errors for either AES or HASH engines
  *
  * \brief \b
  * Description:
  *  This function is used to operate seceral machines one after another 
  *  In order to perform complex cryptographic opeartions 
  *  Supported modes:
  *  Plain >> AES >> AES >> Cipher + HASH(for ciper)
  *  Plain >> AES >> AES >> Cipher + HASH(for Plain)
  *  Plain >> AES >> AES >> Cipher (No HASH)
  *  All modes for each idevidual engine are supported
  *  For detailed description of each of the oprations see the relevant header file
  * \b
  * Algorithm:
  *  -# Make first AES operation
  *  -# Make second AES operation (optionally)
  *  -# Make hash operation
  ***************************************************************/
  CE2Error_t  LLF_AES2AES2HASH_OTF(CE2_AES_IvCounter_t      	FirstIVCounter_ptr,       
    CE2_AES_Key_t           	FirstKey_ptr,                 
    CE2_AES_KeySize_t       	FirstKeySize,            
    CE2_AES_EncryptMode_t   	FirstEncryptDecryptFlag, 
    CE2_AES_OperationMode_t 	FirstOperationMode , 
    CE2_AES_IvCounter_t     	SecondIVCounter_ptr,       
    CE2_AES_Key_t           	SecondKey_ptr,                 
    CE2_AES_KeySize_t       	SecondKeySize,            
    CE2_AES_EncryptMode_t   	SecondEncryptDecryptFlag, 
    CE2_AES_OperationMode_t 	SecondOperationMode ,
    CE2_HASH_ChainMode_t      HASHChainMode,
    CE2_HASH_OperationMode_t	HASHOperationMode,
    DxUint8_t					*DataIn_ptr,        
    DxUint32_t				DataInSize,         
    DxUint8_t					*DataOut_ptr,
    CE2_HASH_Result_t			HashResultBuff);   

  /**
  ****************************************************************
  * Function Name: 
  *  LLF_AES2DES2HASH_OTF
  *
  * Inputs:
  * @param FirstIVCounter_ptr [in] - This parameter is the buffer of the IV or counters on mode CTR for the first key
  *                                  used in the chain.
  *                                  In ECB, XCBC, CMAC mode this parameter is not used.
  *                                  In CBC and MAC modes this parameter should contain the IV values.
  *                                  In CTR mode this value should contain the init counter.
  *                                  In XCBC and CMAC modes it may be NULL
  *  @param FirstKey_ptr [in] - A pointer to the user's key buffer for the first key
  *                                  used in the chain.
  *  @param FirstKeySize [in] - The size of the KEY used by the AES: 128, 192 or 256 bits, 
  *                             as defined in the enum for the first key used in the chain.
  *  @param FirstEncryptDecryptFlag [in] - A flag specifying whether the AES should perform an 
  *                                       Encrypt operation (0) or a Decrypt operation (1). 
  *                                       In XCBC and CMAC modes it must be 0 .
  *                                       for the first key used in the chain.
  *  @param FirstOperationMode [in] - The operation mode: ECB, CBC, MAC, CTR, XCBC (PRF and 96), CMAC.
  *                                   for the first key used in the chain.
  * @param SecondIV_ptr [in] - This parameter is the buffer of the IV (Initial Vector). 
  *                            In ECB mode this parameter is not used and may be NULL.
  *  @param SecondKey_ptr [in] - A pointer to the user's key buffer.
  *                               for the Second key used in the chain.
  *  @param SecondNumOfKeys [in] - The number of used DES keys for the second key used in the chain.
  *  @param SecondEncryptDecryptFlag [in] - A flag specifying whether the DES should perform an 
  *                                          Encrypt operation (0) or a Decrypt operation (1). 
  *                                          for the second key used in the chain.
  *  @param SecondOperationMode [in] - The operation mode: ECB, CBC.
  *                                     for the second key used in the chain.
  * @param HASHChainMode, [in] - The Chain mode - Hash according to enum:
  *                              1. Hash the plain text 
  *                              2. Hash the cipher resulted from the first key
  *                              3. Hash the cipher text resulted
  *                              4. Do not use HASH
  * @param HASHOperationMode [in] - The operation mode - this parameter is ignored in case the chain mode is - No hash
  *                               This parameter is ignored in case the chain mode is - No hash
  *  @param DataIn_ptr [in] - The pointer to the buffer of the input data to the AES chain. 
  *               The pointer's value does not need to be word-aligned.
  *  @param DataInSize [in] - The size of the input data (must be not 0 and must be multiple 
  *               of 16 bytes, besides XCBC and CMAC).
  *  @param DataOut_ptr [out] - The pointer to the buffer of the output data from the AES chain. 
  *                The pointer does not need to be aligned to 32 bits. 
  *                On XCBC and CMAC modes it may be NULL. 
  *                The pointer's value does not need to be word-aligned. 
  * @param HashResultBuff [out] - A pointer to the target buffer where the 
  *                               HASH result stored in the context is loaded to.
  *                               This parameter is ignored in case the chain mode is - No hash
  *
  * Outputs: @returns \b
  *  CE2Error_t  
  *  - comatible errors for either AES, DES or HASH engines
  *
  * \brief \b
  * Description:
  *  This function is used to operate seceral machines one after another 
  *  In order to perform complex cryptographic opeartions 
  *  Supported modes:
  *  Plain >> AES >> DES >> Cipher + HASH(for result cipher)
  *  Plain >> AES >> DES >> Cipher + HASH(for first cipher)
  *  Plain >> AES >> DES >> Cipher + HASH(for Plain)
  *  Plain >> AES >> DES >> Cipher (No HASH)
  *  All modes for each idevidual engine are supported
  *  For detailed description of each of the oprations see the relevant header file
  * \b
  * Algorithm:
  *  -# Make first AES operation
  *  -# Make second DES operation 
  *  -# Make hash operation
  ***************************************************************/
  CE2Error_t  LLF_AES2DES2HASH_OTF(CE2_AES_IvCounter_t      FirstIVCounter_ptr,       
    CE2_AES_Key_t           	FirstKey_ptr,                 
    CE2_AES_KeySize_t       	FirstKeySize,            
    CE2_AES_EncryptMode_t   	FirstEncryptDecryptFlag, 
    CE2_AES_OperationMode_t 	FirstOperationMode ,   
    CE2_DES_Iv_t            	SecondIV_ptr,
    CE2_DES_Key_t           	*SecondKey_ptr,
    CE2_DES_NumOfKeys_t     	SecondNumOfKeys,
    CE2_DES_EncryptMode_t   	SecondEncryptDecryptFlag,
    CE2_DES_OperationMode_t	  SecondOperationMode, 
    CE2_HASH_ChainMode_t      HASHChainMode,
    CE2_HASH_OperationMode_t	HASHOperationMode,
    DxUint8_t					*DataIn_ptr,        
    DxUint32_t				DataInSize,         
    DxUint8_t					*DataOut_ptr,
    CE2_HASH_Result_t			HashResultBuff);

  /**
  ****************************************************************
  * Function Name: 
  *  DX_DES2AES2HASH_OTF
  *
  * Inputs:
  * @param FirstIV_ptr [in] - This parameter is the buffer of the IV (Initial Vector). 
  *                           In ECB mode this parameter is not used and may be NULL.
  *  @param FirstKey_ptr [in] - A pointer to the user's key buffer for the first key
  *                             used in the chain.
  *  @param FirstNumOfKeys [in] - The number of used DES keys for the first key used in the chain.
  *  @param FirstEncryptDecryptFlag [in] - A flag specifying whether the DES should perform an 
  *                                       Encrypt operation (0) or a Decrypt operation (1). 
  *                                       In XCBC and CMAC modes it must be 0 .
  *                                       for the first key used in the chain.
  *  @param FirstOperationMode [in] - The operation mode: ECB, CBC.
  *                                   for the first key used in the chain.
  * @param SecondIVCounter_ptr [in] - This parameter is the buffer of the IV or counters on mode CTR for the first key
  *                                   used in the chain.
  *                                   In ECB, XCBC, CMAC mode this parameter is not used.
  *                                   In CBC and MAC modes this parameter should contain the IV values.
  *                                   In CTR mode this value should contain the init counter.
  *                                   In XCBC and CMAC modes it may be NULL
  *  @param SecondKey_ptr [in] - A pointer to the user's key buffer.
  *                              for the second key used in the chain.
  *  @param SecondKeySize [in] - The size of the KEY used by the AES: 128, 192 or 256 bits, 
  *                              as defined in the enum for the second key used in the chain.
  *  @param SecondEncryptDecryptFlag [in] - A flag specifying whether the AES should perform an 
  *                                          Encrypt operation (0) or a Decrypt operation (1). 
  *                                          for the second key used in the chain.
  *  @param SecondOperationMode [in] - The operation mode: ECB, CBC, MAC, CTR, XCBC (PRF and 96), CMAC.
  *                                    for the second key used in the chain.
  * @param HASHChainMode, [in] - The Chain mode - Hash according to enum:
  *                              1. Hash the plain text 
  *                              2. Hash the cipher resulted from the first key
  *                              3. Hash the cipher text resulted
  *                              4. Do not use HASH
  * @param HASHOperationMode [in] - The operation mode - this parameter is ignored in case the chain mode is - No hash
  *                               This parameter is ignored in case the chain mode is - No hash
  *  @param DataIn_ptr [in] - The pointer to the buffer of the input data to the AES chain. 
  *               The pointer's value does not need to be word-aligned.
  *  @param DataInSize [in] - The size of the input data (must be not 0 and must be multiple 
  *               of 16 bytes, besides XCBC and CMAC).
  *  @param DataOut_ptr [out] - The pointer to the buffer of the output data from the AES chain. 
  *                The pointer does not need to be aligned to 32 bits. 
  *                On XCBC and CMAC modes it may be NULL. 
  *                The pointer's value does not need to be word-aligned. 
  * @param HashResultBuff [out] - A pointer to the target buffer where the 
  *                               HASH result stored in the context is loaded to.
  *                               This parameter is ignored in case the chain mode is - No hash
  *
  * Outputs: @returns \b
  *  CE2Error_t  
  *  - comatible errors for either AES, DES or HASH engines
  *
  * \brief \b
  * Description:
  *  This function is used to operate seceral machines one after another 
  *  In order to perform complex cryptographic opeartions 
  *  Supported modes:
  *  Plain >> DES >> AES >> Cipher + HASH(for result cipher)
  *  Plain >> DES >> AES >> Cipher + HASH(for first cipher)
  *  Plain >> DES >> AES >> Cipher + HASH(for Plain)
  *  Plain >> DES >> AES >> Cipher (No HASH)
  *  All modes for each idevidual engine are supported
  *  For detailed description of each of the oprations see the relevant header file
  * \b
  * Algorithm:
  *  -# Make first DES operation
  *  -# Make second AES operation 
  *  -# Make hash operation
  ***************************************************************/
  CE2Error_t  LLF_DES2AES2HASH_OTF(CE2_DES_Iv_t            	FirstIV_ptr,
    CE2_DES_Key_t           	*FirstKey_ptr,
    CE2_DES_NumOfKeys_t     	FirstNumOfKeys,
    CE2_DES_EncryptMode_t   	FirstEncryptDecryptFlag,
    CE2_DES_OperationMode_t 	FirstOperationMode,  
    CE2_AES_IvCounter_t     	SecondIVCounter_ptr,       
    CE2_AES_Key_t           	SecondKey_ptr,                 
    CE2_AES_KeySize_t       	SecondKeySize,            
    CE2_AES_EncryptMode_t   	SecondEncryptDecryptFlag, 
    CE2_AES_OperationMode_t 	SecondOperationMode ,
    CE2_HASH_ChainMode_t      HASHChainMode,
    CE2_HASH_OperationMode_t	HASHOperationMode,
    DxUint8_t					*DataIn_ptr,        
    DxUint32_t				DataInSize,         
    DxUint8_t					*DataOut_ptr,
    CE2_HASH_Result_t			HashResultBuff);

#ifdef __cplusplus
}
#endif

#endif /* LLF_OTF_H */
